import re
import os


def format_path(path):
    ''' Replace any sequence of \\ and|or / by a unique / in the given path (string).

    Raise:
        AttributeError: if path is empty or not a string
    '''

    if not isinstance(path, str):
        raise AttributeError("Given path need to be a string.")

    path = path.replace('\\', '/')

    if path == "":
        raise AttributeError("Given path need to be a string.")

    while path.find('//') != -1:
        path = path.replace('//', '/')

    return path

def format_folder_path(path):
    ''' Replace any sequence of \\ and|or / by a unique / in the given path (string).
    And add a trailing / at the end of the path.

    Raise:
        AttributeError: if path is empty or not a string
    '''

    path = format_path(path)

    if path[-1] != '/':
        return path + '/'

    return path

def format_file_path(path):
    '''Replace any sequence of \\ and|or / by a unique / in the given path (string).

    Raise:
        AttributeError: if path is empty or not a string or if the path ends by /.
    '''

    path = format_path(path)

    if path[-1] == "/":
        raise AttributeError(f"Provided file path ends with /: {path}")
    return path

def split_filePath(path, secondaryExt=""):
    """
    From a file path as string, extract the folder path, the file name, the main extension and the secondary extensions.

    If not secondaryExt provided, return empty string
    If no main extension found, return empty string
    If secondaryExt provided not found, raise AttributeError

    For example: split_filePath("/path/to/somefile.second.vi.ext", "second.vi") returns ('/path/to/', 'somefile', '.ext', '.second.vi')
    """

    if not isinstance(secondaryExt, str):
        raise AttributeError(f"Wrong attribut for extensions, should be string you provided {type(secondaryExt)}")

    path = format_file_path(path)

    folderPath = os.path.dirname(path)
    if folderPath != "":
        folderPath = folderPath + "/"

    filename = path.replace(folderPath, "")
    filename, mainExt = os.path.splitext(filename)

    if not secondaryExt:
        return folderPath, filename, mainExt, ''

    if not secondaryExt in filename:
        raise AttributeError(f"File of path {path} doesn't have secondary extension expected: {secondaryExt}")

    if not secondaryExt.startswith('.'):
        secondaryExt = "." + secondaryExt

    filename = filename.replace(secondaryExt, "")

    return folderPath, filename, mainExt, secondaryExt