import json
import re
import os

from files_folders.extensions.AbstractExtension import AbstractExtension
from files_folders.File import _deleted_check


class JsonFile(AbstractExtension):
    ''' Represente a json file. Provide everything you need to handle the file.
    Allow you to write and read the json as a dict.

    Args:
        path (string): full file path
        createIfNotFound (False): to create the file if it doesn't exist.
        createFolderIfNotFound (False): to create the containing folder and its parents folders if they don't exist.
    '''

    def __init__(self, path, secondaryExt = "", createIfNotFound=False, createFolderIfNotFound=False):
        super().__init__(path, mainExtension="json", secondaryExt=secondaryExt, createIfNotFound=createIfNotFound, createFolderIfNotFound=createFolderIfNotFound)

    @_deleted_check
    def read(self):
        ''' Read all at once (warning for big files!!).
        Return a dict.

        Raise:
            IOError: if the file isn't readable
        '''

        file = open(self.get_filePath(), 'r')
        try:
            content = json.load(file)
        except:
            raise IOError(f"Can't read this Json file: {self.get_filePath()}. Perhaps the content is badly formatted.")
        file.close()

        return content

    @_deleted_check
    def write(self, content):
        ''' Overwrite the file with the given content.

        Args:
            content (dict)
        '''

        if not isinstance(content, dict):
            raise AttributeError(f"Content need to be a dict. Can't write file {self.get_filePath()}")

        with open(self.get_filePath(), "w") as file:
            json.dump(content, file)

        return self

    @_deleted_check
    def write_keyValue(self, key, value, replaceIfExist=True, addIfNotFound=True):
        ''' Change or Add a key and its value in the current json dict.

        Args:
            key
            value
            replaceIfExist (True): replace the value of this key if exist.

        Raise:
            AttributeError: if key doesn't exist and addIfNotFound is setted to False.
            AttributeError: if key exist and replaceIfExist is False.

        '''

        content = self.read()

        if not replaceIfExist and key in content:
            raise AttributeError(f"Key {key} already exist in this json file: {self.get_filePath()}. Set replaceIfExist to True if you cant to replace it.")

        if not addIfNotFound and not key in content:
            raise AttributeError(f"No key {key} in the json file: {self.get_filePath()}. Set addIfNotFound to True if you want to add it.")

        content[key] = value
        self.write(content)

        return self
