import os
import shutil
import re
import functools

from files_folders import utils
from files_folders.exceptions import *


# Decorators
def _deleted_check(func):
    ''' If this instance of Folder is linked to an inexisting folder, raise an error.'''

    @functools.wraps(func)
    def wrapper(*args, **kwargs):

        folder = args[0]

        if not os.path.isdir(folder.path):
            raise FolderNotFoundError(folder.path, "You can't use Folder class if the folder it represents doesn't exist. This folder was probably deleted.")
        
        return func(*args, **kwargs)
    return wrapper


# Class
class Folder():
    ''' Represente an existing folder.

    Args:
        path (string)
        createIfNotFound (False): If target folder doesn't exist, create it (and its parents folders if needed).
     
    Raise:
        FolderNotFoundError: If no folder founded for this path and createIfNotFound is False.
    '''
    
    def __init__(self, path, createIfNotFound=False):
        self.path = utils.format_folder_path(path)
        
        if not os.path.isdir(self.path):
            if createIfNotFound:
                os.makedirs(path)
            else:
                raise FolderNotFoundError(self.path, "Can't create a new instance of Folder.")
        
    # Getters
    def get_path(self):
        return self.path

    def get_folder_name(self):
        ''' Example: for something/anotherthing/folderName/ returns folderName/
        
        Raise:
            AttributeError: If no folderName founded (not supposed to happend).
        '''

        research = re.findall(r".*\/", self.path[:-1])
        
        if not research:
            if self.path[-1] == '/':
                return self.path
            else:
                raise AttributeError(f"No folder name found for {self.path}")
        else:
            return self.path.replace(str(research[-1]), "")        

    def get_parent_folder(self):
        ''' Example: for something/anotherthing/folderName/ returns something/anotherthing/ '''

        return self.path[:self.path.rfind(self.get_folder_name())]


    # Functions
    @_deleted_check
    def rename(self, newName, replaceIfExist=False):
        ''' Rename the folder. 
        
        Args:
            newName (string): can contain trailing / or not.
            replaceIfExist (False): if True replace the existing folder by this one.

        Raise: 
            AttributeError: if more than the ending / was found in newName.
            FolderExistsError: if target folder already exist and replaceIfExist is False.
        '''

        newName = utils.format_folder_path(newName)
        
        if len(re.findall(r"/", newName)) != 1:
            raise AttributeError(f"Can't rename this folder: {self.path}.\nNew name badly formatted: / was found not only at the end of: {newName}.")
    
        folderPath = self.get_parent_folder() + newName

        if os.path.isdir(folderPath):
            if replaceIfExist:
                shutil.rmtree(folderPath)
                shutil.move(self.path, folderPath)
                self.path = folderPath
                return self
            else:
                raise FolderExistsError(folderPath, f"Can't rename this folder: {self.path}")
                
        shutil.move(self.path, folderPath)
        self.path = folderPath
        return self


    @_deleted_check
    def move_in(self, targetFolderPath, replaceIfExist=False, createParentsFoldersIfNotFound=False):
        ''' Move this folder an its content into the target folder.

        Args:
            targetFolderPath (string): containing folder target (but not the current foldername)
            replaceIfExist (False): if the target folder already exist, replace it;
            createParentsFoldersIfNotFound (False)
        
        Raise:
            FolderExistsError: if target folder already exist and replaceIfExist is false.
            FolderNotFoundError: if target parent folder doesn't exist and createParentsFoldersIfNotFound False. '''

        targetFolderPath = utils.format_folder_path(targetFolderPath)
        folderPath = targetFolderPath + self.get_folder_name()

        if os.path.isdir(folderPath):
            if replaceIfExist:
                shutil.rmtree(folderPath)
                shutil.move(self.path, folderPath)
                self.path = folderPath
                return self
            else:
                raise FolderExistsError(folderPath, f"Can't move this folder: {self.path}")
        elif not os.path.isdir(targetFolderPath):
            if createParentsFoldersIfNotFound:
                os.makedirs(targetFolderPath)
            else:
                raise FolderNotFoundError(targetFolderPath, f"Can't move this folder: {self.path}")

        shutil.move(self.path, folderPath)
        self.path = folderPath
        return self

    @_deleted_check
    def delete(self):
        ''' Delete the folder and its content. '''
        shutil.rmtree(self.path, ignore_errors=True)

    @_deleted_check
    def get_files(self):
        ''' Get the list of files name in this folder. (only file names, no file path in them) '''

        filesList = []

        for f in os.listdir(self.path):
            f = utils.format_file_path(f)

            if os.path.isfile(os.path.join(self.path, f)):
                filesList.append(f)

        return filesList

    @_deleted_check
    def get_subfolders(self):
        ''' Get the list of folder name in this folder. (only folder names, no folder path in them) '''

        foldersList = []

        for f in os.listdir(self.path):
            f = utils.format_folder_path(f)

            if os.path.isdir(os.path.join(self.path, f)):
                foldersList.append(f)

        return foldersList

    @_deleted_check
    def is_empty(self):
        ''' Check if this folder doesn't contain file or folder. '''

        return len(self.get_files() + self.get_subfolders()) == 0

    @_deleted_check
    def delete_files(self):
        ''' Delete all files in the directory and return the list of these files names. '''

        deleted_files = []
        for item in os.listdir(self.path):
            item = utils.format_file_path(item)
            if os.path.isfile(self.path + item):
                deleted_files.append(item)
                os.remove(self.path + item)
        return deleted_files

    @_deleted_check
    def delet_subfolders(self):
        ''' Delete all folders in this directory and return the list of these folders names. '''

        deleted_dir = []
        for item in os.listdir(self.path):
            item = utils.format_folder_path(item)
            if os.path.isdir(self.path + item):
                deleted_dir.append(item)
                shutil.rmtree(self.path + item)
        return deleted_dir
    
    @_deleted_check
    def clear(self):
        ''' Delete files and folders of this directory.
        Return (list of these files names, list of these folders names). '''

        return (self.delete_files(), self.delet_subfolders())
    
    @_deleted_check
    def copy(self):
        '''Copy this folder by adding (X) at the end of the folderName (X is a number).
        Return the Folder object associated. '''

        index = 1
        while os.path.isdir(self.path[:-1] + "(" + str(index) + ")" + "/"):
            index += 1

        path = self.path[:-1] + "(" + str(index) + ")" + "/"
        shutil.copytree(self.path, path)

        return Folder(path, createIfNotFound=False)

    @_deleted_check
    def copy_to_folder(self, targetFolderPath, replaceIfExist=False, createParentFolderIfNotFound=False):
        ''' Copy this folder into the target folderPath.
        If target is the current parrent folder, use copy function instead.

        Return the Folder object of this copy.

        Args:
            targetFolderPath (string): the parent folder path.
            replaceIfExist (False): replace target folder if exist
            createParentFolderIfNotFound (False): create parent folder if needed
        
        Raise:
            FolderNotFoundError: if parent folder doesn't exist and createParentFolderIfNotFound is False
            FolderExistsError: if target folder exist and replaceIfExist is false

        Note:
            Different from merge in because will never keep the target content.
        '''

        targetFolderPath = utils.format_folder_path(targetFolderPath)

        if targetFolderPath == self.get_parent_folder():
            return self.copy()

        if not os.path.isdir(targetFolderPath):
            if createParentFolderIfNotFound:
                os.makedirs(targetFolderPath)
            else:
                raise FolderNotFoundError(targetFolderPath, f"No parent folder for this copy. Can't copy the current folder: {self.path}")

        folderPath = targetFolderPath + self.get_folder_name()

        if os.path.isdir(folderPath):
            if replaceIfExist:
                shutil.rmtree(folderPath)
            else:
                raise FolderExistsError(folderPath, f"Can't copy the current folder: {self.path}")

        shutil.copytree(self.path, folderPath)

        return Folder(folderPath, createIfNotFound=False)        
        
    def merge_in(self, folderPath, replaceIfExist=False, createIfNotFound=False):
        ''' Merge the content of this folder in the given folder (without changing its name).

        Return the Folder object of the target folder.

        Args:
            folderPath (string): folder to merge in
            replaceIfExist (False): False means that if two files or two sub folders have the same name, keep the original. True for the new items.
            createIfNotFound (False)

        Raise:
            FolderNotFoundError: if no target folder and createIfNotFound is false

        Warning:
            do not manage deeper merge. If you have /origin/some/file.txt  and /current/some/other.txt you will not keep these two files into you folder origin.

        Note:
            Will not delete the current folder.
        '''

        folderPath = utils.format_folder_path(folderPath)

        if not os.path.isdir(folderPath):
            if createIfNotFound:
                os.makedirs(folderPath)
            else:
                raise FolderNotFoundError(folderPath, f"Can't merge this folder in it: {self.path}")
    
        for filePath in self.get_files():
            newFilePath = folderPath + filePath
            filePath = self.path + filePath
            if os.path.isfile(newFilePath):
                if replaceIfExist:
                    os.remove(newFilePath)
                    shutil.copyfile(filePath, newFilePath)
            else:
                shutil.copyfile(filePath, newFilePath)
        
        for subfolderPath in self.get_subfolders():
            newSubfolderPath = folderPath + subfolderPath
            subfolderPath = self.path + subfolderPath
            if os.path.isdir(newSubfolderPath):
                if replaceIfExist:
                    shutil.rmtree(newSubfolderPath, ignore_errors=True)
                    shutil.copytree(subfolderPath, newSubfolderPath)
            else:
                shutil.copytree(subfolderPath, newSubfolderPath)
        
        return Folder(folderPath)
                    


                





