import os
import shutil
import re
import functools
import subprocess

from files_folders import utils
from files_folders.exceptions import *
from files_folders.AbstractFile import AbstractFile, _deleted_check, _convert_extensions_list_to_string, _convert_extensions_string_to_list, _various_extensions_input_to_list

# Class
class File(AbstractFile):


    def __init__(self, path, secondaryExt = "", createIfNotFound=False, createFolderIfNotFound=False):
        # Docstring du parent
        super().__init__(path, secondaryExt=secondaryExt, createIfNotFound=createIfNotFound, createFolderIfNotFound=createFolderIfNotFound)

    @_deleted_check
    def change_extensions(self, newExtensions, replaceIfExist=False):
        ''' Change the file extensions.

        Args:
            newExtensions (string or list)
            replaceIfExist (False): replace if a file with the target name exist.

        Raise:
            FileExistsError: if target exist and replaceIfExist False
            AttributeError: if empty arg was provided
        '''

        newExtensions = _various_extensions_input_to_list(newExtensions)

        if newExtensions == []:
            raise AttributeError("Given extensions is empty") # Un comment comme ca pour tester

        filePath = self.folderPath + self.fileName + _convert_extensions_list_to_string(newExtensions)

        # Ici c'est structurel
        if os.path.isfile(filePath):
            if replaceIfExist:
                os.remove(filePath)
            else:
                raise FileExistsError(filePath, f"Can't change file extensions of: {self.get_filePath()}")

        os.replace(self.get_filePath(), filePath)
        self.extensions = newExtensions
        return self

    @_deleted_check
    def change_main_extension(self, newExtension, replaceIfExist=False):
        ''' Change the file main extensions.

        Args:
            newExtension (string): with or without dot
            replaceIfExist (Flase): replace if a file with the target name exist.

        Raise:
            FileExistsError: file exist and replaceIfExist False
            AttributeError: wrong format (more than one dot in the given string or empty string)
        '''

        # Remove the point at the beggining of the extension
        if newExtension == "" or newExtension == '.':
            raise AttributeError(f"Extension provided is empty.")
        elif newExtension[0] == '.':
            newExtension = newExtension[1:]

        if len(re.findall(r"\.", newExtension)) > 0:
            raise AttributeError(f"To much dots.")

        if self.has_secondary_extensions():
            newExtension = self.extensions[:-1] + [newExtension]

        self.change_extensions(newExtension, replaceIfExist=replaceIfExist)
        return self

    @_deleted_check
    def remove_extensions(self, replaceIfExist=False):
        ''' Delete all extensions of the file.

        Raise:
            FileExistsError: if target exist and replaceIfExist False '''

        if self.extensions != []:
            filePath = self.folderPath + self.fileName

            if os.path.isfile(filePath):
                if replaceIfExist:
                    os.remove(filePath)
                else:
                    raise FileExistsError(filePath, f"Can't remove extensions of: {self.get_filePath()}")

            os.replace(self.get_filePath(), filePath)
            self.extensions = []
        return self

    @_deleted_check
    def remove_main_extension(self, replaceIfExist=False):
        ''' Delete main extension of the file.

        Raise:
            FileExistsError: if target exist and replaceIfExist False '''

        if self.extensions:
            secondaryExtensions = []
            if self.has_secondary_extensions():
                secondaryExtensions = self.extensions[:-1]
                filePath = self.folderPath + self.fileName + _convert_extensions_list_to_string(secondaryExtensions)
            else:
                filePath = self.folderPath + self.fileName

            if os.path.isfile(filePath):
                if replaceIfExist:
                    os.remove(filePath)
                else:
                    raise FileExistsError(filePath, f"Can't remove main extensions of: {self.get_filePath()}")

            os.replace(self.get_filePath(), filePath)

            self.extensions = secondaryExtensions
        return self

    @_deleted_check
    def copy(self):
        '''
        Return a File object of this copied file.
        '''
        #DocString du parent

        filePath = super().copy()

        return File(filePath, createIfNotFound=False, createFolderIfNotFound=False)

    @_deleted_check
    def copy_to_folder(self, folderPath, replaceIfExist=False, createFolderIfNotFound=False):
        '''
        Return a File object of this copied file.
        '''
        # DocString du parent

        filePath = super().copy_to_folder(folderPath, replaceIfExist=replaceIfExist, createFolderIfNotFound=createFolderIfNotFound)

        return File(filePath, createIfNotFound=False, createFolderIfNotFound=False)

    @_deleted_check
    def write_at_end(self, content):
        ''' Write the givent content at the end of the file.
        Warning: not suited for all kinf of file. If needed use the specifique file extension class.'''

        with open(self.get_filePath(), "a") as file:
            file.write(content)

        return self

    @_deleted_check
    def write_at_beginning(self, content):
        ''' Add the given content at the beginning of the file.
        Warning: Not optimised because need to copy all the file content (if to big: memory issue).
        Warning: not suited for all kinf of file. If needed use the specifique file extension class.'''

        currentContent = self.read()
        self.write(content + '\n' + currentContent)
        return self

    @_deleted_check
    def open_in_default_app(self):
        """Open this file with the software
        with which one your computer is supposed to open this kind of file.

        Returns:
            <self>
        """

        if sys.platform == "win32":
            os.startfile(filename)
        else:
            opener ="open" if sys.platform == "darwin" else "xdg-open"
            subprocess.call([opener, self.get_filePath])

        return self

