import os
import shutil
import re
import functools

from files_folders import utils
from files_folders.exceptions import *


# Decorators
def _deleted_check(func):
    ''' If this instance of File is linked to an inexisting file, raise an error.'''

    @functools.wraps(func)
    def wrapper(*args, **kwargs):

        file = args[0]

        if not os.path.isfile(file.get_filePath()):
            raise FileNotFoundError(file.get_filePath(), "You can't use File class if the file it represents doesn't exist. This file was probably deleted.")

        return func(*args, **kwargs)
    return wrapper

# Tools
def _convert_extensions_string_to_list(extensions):
    ''' Take .txt.json and return ['txt', 'json'].
    For empty string return emtpy list.
    Starting point can be omited (if ending point will be ignored)
    Multiple points will be considered as only one point.
    '''

    return [ext for ext in extensions.split('.') if ext != '']

def _convert_extensions_list_to_string(extensions):
    ''' Take ['txt', 'json'] and return .txt.json.
    No point inside the list. '''

    return '.' + '.'.join(extensions)

def _various_extensions_input_to_list(extensions):
    ''' Convert the entry into a list of extensions without dot ["txt", "jsons].
    inputs can be:
    string: ".txt.json", starting point can be ommited. If trailing point will be ommited.
    string with many points: ".txt..json", replace by one dot and convert
    empty string: return empty list
    list with mixed dots and non dots: [".txt.", "json"]: remove all dots

    '''
    if isinstance(extensions, list):
        for k in range(len(extensions)):
            if extensions[k][0] == '.':
                extensions[k] = extensions[k][1:]
            if extensions[k][-1] == '.':
                extensions[k] = extensions[k][:-1]
    elif isinstance(extensions, str):
        extensions = _convert_extensions_string_to_list(extensions)

    return extensions


# Class
class AbstractFile():

    def __init__(self, path, secondaryExt = "", createIfNotFound=False, createFolderIfNotFound=False):
        ''' Represente an existing file.

        Attributs:
            folderPath (string): path to the containing folder
            fileName (string): without extensions
            extensions (list): list of extensions as sting, without point

        Args:
            path (string): full file path
            createIfNotFound (False): to create the file if it doesn't exist.
            createFolderIfNotFound (False): to create the containing folder and its parents folders if they don't exist.

        Raise:
            FileNotFoundError: if no file and createIfNotFound False.
            FolderNotFoundError: if no containing folder and createFolderIfNotFound False.
        '''

        folderPath, fileName, mainExt, secondaryExt = utils.split_filePath(path, secondaryExt)

        if folderPath != "" and not os.path.isdir(folderPath):
            if createFolderIfNotFound:
                os.makedirs(folderPath)
            else:
                raise FolderNotFoundError(folderPath, f"Can't instanciate this File: {path}, containing folder doesn't exist.")

        if not os.path.isfile(path):
            if createIfNotFound:
                with open(path, 'w'):
                    pass
            else:
                raise FileNotFoundError(path, f"Can't instanciate this File: {path}")

        self.folderPath = folderPath
        self.fileName = fileName
        self.extensions = _convert_extensions_string_to_list(secondaryExt + mainExt)

    # Getters (no setter)
    def get_folderPath(self):
        return self.folderPath

    def get_fileName(self):
        return self.fileName

    def get_extensions_list(self):
        return self.extensions

    def get_secondary_extensions_list(self):
        secondaryExt = self.get_extensions_list().copy()
        if len(secondaryExt) > 1:
            del secondaryExt[-1]
            return secondaryExt
        else:
            return []

    def get_secondary_extensions_string(self):
        secondaryExt = _convert_extensions_list_to_string(self.get_secondary_extensions_list())
        if secondaryExt == ".":
            return ""
        return secondaryExt

    # Complexe getter
    def get_filePath(self):
        ''' Return the full file path. '''
        return self.folderPath + self.get_fileName_with_extensions()

    def get_fileName_with_secondary_extensions(self):
        ''' Return the file name with its secondary extensions'''

        if len(self.extensions) >= 2:
            return self.fileName + self.get_extensions_string()[:-len(self.get_main_extension())]

        return self.fileName

    def get_fileName_with_extensions(self):
        ''' Return the file name with its extensions'''
        return self.fileName + self.get_extensions_string()

    def get_extensions_string(self):
        ''' Return string like: .myExt2.myExt1.myMainExt. '''
        if self.extensions == []:
            return ""
        return _convert_extensions_list_to_string(self.extensions)

    def get_main_extension(self):
        ''' Return the main extension with a dot. Empty string if not.'''
        if not self.extensions:
            return ""
        return '.' + self.extensions[-1]

    # Functions
    @_deleted_check
    def delete(self):
        ''' Delete this file. '''
        os.remove(self.get_filePath())

    @_deleted_check
    def move_to_folder(self, folderPath, replaceIfExist=False, createFolderIfNotFound=False):
        ''' Move this file to the given folder.

        Args:
            folderPath (string)
            replaceIfExist (True): replace the existing file with the same name in the target location.
            createFolderIfNotFound (False): if target folder doesn't exist (parents also if needed).

        Raise:
            FolderNotFoundError: No folder at target location and createFolderIfNotFound False
            FileExistsError: if file already exist at target location and replaceIfExist False
        '''

        folderPath = utils.format_folder_path(folderPath)
        filePath = folderPath + self.get_fileName_with_extensions()

        if not os.path.isdir(folderPath):
            if createFolderIfNotFound:
                os.makedirs(folderPath)
            else:
                raise FolderNotFoundError(folderPath, f"Can't move this file: {self.get_filePath()}")

        if os.path.isfile(filePath):
            if replaceIfExist:
                os.remove(filePath)
            else:
                raise FileExistsError(filePath, f"Can't move this file: {self.get_filePath()}")

        os.replace(self.get_filePath(), filePath)
        self.folderPath = folderPath
        return self

    @_deleted_check
    def rename_no_extension_change(self, fileName, replaceIfExist=False):
        '''Doesn't change extensions.

        Args:
            fileName (string)
            replaceIfExist (False): replace if a file with the target name exist.

        Raise:
            AttributeError: emtpy string provided or new name contains too much /
            FileExistsError: file exist at target location and replaceIfExist False
        '''

        fileName = utils.format_file_path(fileName)
        if fileName.find("/") != -1:
            raise AttributeError(f"The new name for this file was not good. Folder inside: {fileName}")

        extensions = self.get_extensions_string()

        folderPath, fileName, mainExt, secondaryExt = utils.split_filePath(fileName + extensions, secondaryExt=self.get_secondary_extensions_string())
        extensions = secondaryExt + mainExt

        filePath = self.folderPath + fileName + extensions

        if filePath == self.get_filePath():
            return self

        if os.path.isfile(filePath):
            if replaceIfExist:
                os.remove(filePath)
            else:
                raise FileExistsError(filePath, f"Can't rename this file: {self.get_filePath()}")

        os.replace(self.get_filePath(), filePath)

        self.fileName = fileName
        if extensions:
            self.extensions = _convert_extensions_string_to_list(extensions)

        return self

    @_deleted_check
    def rename(self, fileName, secondaryExt = "", replaceIfExist=False):
        ''' Can contains extensions. Can't be empty string.
        Change filename with extensions.

        Args:
            fileName (string): new filename
            secondaryExt (string): contains secondary extensions included in the new fileName, for example '.second.ext' or 'second.ext'
            replaceIfExist (False): replace if a file with the target name exist.

        Raise:
            AttributeError: emtpy string provided or new name contains too much /
            FileExistsError: file exist at target location and replaceIfExist False
        '''

        fileName = utils.format_file_path(fileName)
        if fileName.find("/") != -1:
            raise AttributeError(f"The new name for this file was not good. Folder inside: {fileName}")

        folderPath, fileName, mainExt, secondaryExt = utils.split_filePath(fileName, secondaryExt=secondaryExt)
        extensions = secondaryExt + mainExt

        filePath = self.folderPath + fileName + extensions

        if filePath == self.get_filePath():
            return self

        if os.path.isfile(filePath):
            if replaceIfExist:
                os.remove(filePath)
            else:
                raise FileExistsError(filePath, f"Can't rename this file: {self.get_filePath()}")

        os.replace(self.get_filePath(), filePath)

        self.fileName = fileName
        if extensions:
            self.extensions = _convert_extensions_string_to_list(extensions)

        return self

    @_deleted_check
    def change_secondary_extensions(self, newExtensions, replaceIfExist=False):
        ''' Change the file secondary extensions.

        Args:
            newExtension (string or list)
            replaceIfExist (Flase): replace if a file with the target name exist.

        Raise:
            FileExistsError: file exist and replaceIfExist False
            AttributeError: empty input provided
        '''

        if not newExtensions or newExtensions == ".":
            raise AttributeError("Empty argument done")

        if self.extensions == []:
            raise AttributeError(f"Can't change secondary extensions, there is no primary extension for: {self.get_filePath()}")

        newExtensions = _various_extensions_input_to_list(newExtensions)
        if newExtensions == []:
            raise AttributeError("Given extensions is empty")

        newExtensions = newExtensions + [self.extensions[-1]]

        filePath = self.folderPath + self.fileName + _convert_extensions_list_to_string(newExtensions)

        if os.path.isfile(filePath):
            if replaceIfExist:
                os.remove(filePath)
            else:
                raise FileExistsError(filePath, f"Can't change file extensions of: {self.get_filePath()}")

        os.replace(self.get_filePath(), filePath)
        self.extensions = newExtensions
        return self

    @_deleted_check
    def has_secondary_extensions(self, replaceIfExist=False):
        return len(self.extensions) > 1

    @_deleted_check
    def remove_secondary_extensions(self, replaceIfExist=False):
        ''' Delete all secondary extensions of the file.

        Raise:
            FileExistsError: if target exist and replaceIfExist False '''

        if self.has_secondary_extensions():
            mainExt = ""
            if self.extensions:
                mainExt = self.extensions[-1]

            filePath = self.folderPath + self.fileName + '.' + mainExt

            if os.path.isfile(filePath):
                if replaceIfExist:
                    os.remove(filePath)
                else:
                    raise FileExistsError(filePath, f"Can't remove secondary extensions of: {self.get_filePath()}")

            os.replace(self.get_filePath(), filePath)

            self.extensions = []
            if mainExt:
                self.extensions.append(mainExt)

        return self

    @_deleted_check
    def read(self):
        ''' Read all in at once (warning for big files!!). '''

        file = open(self.get_filePath(), 'r')
        content = file.read()
        file.close()
        return content

    def write(self, content):
        ''' Overwrite the file with the given content. '''

        with open(self.get_filePath(), "w") as file:
            file.write(content)

        return self

    @_deleted_check
    def copy(self):
        ''' Copy this file. The copied file name will be followed by: (X), with X a number.

        Return the copied filePath
        '''

        folderPath = utils.format_folder_path(self.folderPath)
        filePathWithoutExt = folderPath + self.fileName

        index = 1

        while os.path.isfile(filePathWithoutExt + "(" + str(index) + ")" + self.get_extensions_string()):
              index += 1

        filePath = filePathWithoutExt + "(" + str(index) + ")" + self.get_extensions_string()

        shutil.copyfile(self.get_filePath(), filePath)

        return filePath


    @_deleted_check
    def copy_to_folder(self, folderPath, replaceIfExist=False, createFolderIfNotFound=False):
        ''' Copy this file into the target folderPath. Keep the same file.

        Args:
            folderPath (string): target containing folder
            replaceIfExist (False)
            createFolderIfNotFound (False)

        Return the copied filedPath

        Raise:
            FolderNotFoundError: if target containing folder not found and createFolderIfNotFound False
            FileExistsError: if target file already exist and replaceIfExist False

        Note:
            If target is the current folder, use copy function instead.
        '''

        if folderPath == self.folderPath:
            return self.copy()

        folderPath = utils.format_folder_path(folderPath)
        filePath = folderPath + self.get_fileName_with_extensions()

        if not os.path.isdir(folderPath):
            if createFolderIfNotFound:
                os.makedirs(folderPath)
            else:
                raise FolderNotFoundError(folderPath, "No containing folder.")

        if os.path.isfile(filePath):
            if replaceIfExist:
                os.remove(filePath)
            else:
                raise FileExistsError(filePath, "Target file already exist")

        shutil.copyfile(self.get_filePath(), filePath)

        return filePath
